/*******************************************************************************
 * Copyright (c) 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.ui.internal.themes;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;

import org.eclipse.jface.resource.StringConverter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Display;


/**
 * Useful color utilities.
 * 
 * @since 3.0
 */
public final class ColorUtils {
	
	/**
	 * @param value the SWT constant <code>String</code>.
	 * @return the value of the SWT constant, or <code>SWT.COLOR_BLACK</code> if it could 
	 * not be determined.
	 */
	private static RGB process(String value) {
		try {
			Class clazz = SWT.class; //$NON-NLS-1$
			Field[] fields = clazz.getDeclaredFields();
			for (int i = 0; i < fields.length; i++) {
				Field field = fields[i];
				if (field.getType() == Integer.TYPE
					&& Modifier.isStatic(field.getModifiers())
					&& Modifier.isPublic(field.getModifiers())
					&& Modifier.isFinal(field.getModifiers())) {
					if (value.equals(field.getName())) {
						return getSystemColor(field.getInt(null));
					}
				}
			}
		} catch (IllegalArgumentException e) {
			// no op - shouldnt happen.  We check for static before calling getInt(null)
		} catch (IllegalAccessException e) {		    
			// no op - shouldnt happen.  We check for public before calling getInt(null)
		}
		return getSystemColor(SWT.COLOR_BLACK);
	}	

	/**
	 * Blend the two color values returning a value that is halfway between them.
	 * 
	 * @param val1 the first value
	 * @param val2 the second value
	 * @return the blended color
	 */
	public static RGB blend(RGB val1, RGB val2) {	
		int red = blend(val1.red, val2.red);
		int green = blend(val1.green, val2.green);
		int blue = blend(val1.blue, val2.blue);		
		return new RGB(red, green, blue);
	}
	
	/**
	 * Blend the two color values returning a value that is halfway between them.
	 * 
	 * @param temp1 the first value
	 * @param temp2 the second value
	 * @return the blended int value
	 */
	private static int blend(int temp1, int temp2) {
		return (Math.abs(temp1 - temp2) / 2) + Math.min(temp1,temp2);
	}
	

	/**
	 * @param colorId the system color identifier.
	 * @return the RGB value of the supplied system color.
	 */
	private static RGB getSystemColor(int colorId) {
		return Display.getCurrent().getSystemColor(colorId).getRGB();
	}	

	/**
	 * Get the RGB value for a given color.
     * 
     * @param rawValue the raw value, either an RGB triple or an SWT constant.
     * @return Returns the RGB value.
	 */
    public static RGB getColorValue(String rawValue) {
		if (rawValue == null)
			return null;
		
		rawValue = rawValue.trim();

		if (!isDirectValue(rawValue)) {
			return process(rawValue);
		}

		return StringConverter.asRGB(rawValue);        
    }

    /**
	 * Get the RGB values for a given color array.
     * 
     * @param rawValues the raw values, either RGB triple or an SWT constant.
     * @return Returns the RGB values.
	 */
    public static RGB [] getColorValues(String[] rawValues) {
    	RGB [] values = new RGB[rawValues.length];
    	for (int i = 0; i < rawValues.length; i++) {
            values[i] = getColorValue(rawValues[i]);
        }
    	return values;
    }

    /** 
	 * @return whether the value returned by <code>getValue()</code> is already 
	 * in RGB form. 
	 */
	private static boolean isDirectValue(String rawValue) { 
		return rawValue == null ? true : rawValue.matches("\\d{1,3},\\d{1,3},\\d{1,3}"); //$NON-NLS-1$
	}
    
    /**
     * Not intended to be instantiated.
     */
    private ColorUtils() {
        //no-op
    }
}
